function [t,x] = ieuler(field, t0, T, x0, h, tol)
%IEULER Implements the Implicit Euler one-step ODE solver
%
% Parameters:
%  field  -- Right hand side function of ODE system: x'=f(t,x)
%  t0     -- Initial time
%  T      -- End time (T > t0)
%  x0     -- Initial value
%  h      -- Size of time step (h <= T-t0)
%  tol    -- Tolerance to use for solving the fixed point iteration (default=0.01)
%
% Outputs:
%  t  -- [t0; t-0+h, t0+2*h; ...; t0+i*h; ...]
%  x  -- Matrix containing numerical solution, with each row the value of x
%        at each time step

arguments
    field; t0; T; x0; h;
    tol = 0.01;
end

n = ceil((T-t0)/h);
t = t0+h*(0:n).';
x = ones(n+1,length(x0));
x(1,:) = x0;

for i=1:n
    k1_old = feval(field, t(i), x(i,:).'); % k1_old = k1^{(0)}
    k1_new = feval(field, t(i)+h, x(i,:).'+h*k1_old); % k1_new = k1^{(1)}
    while norm(k1_old-k1_new, 'inf') >= tol
        k1_old = k1_new; % k1^{(n)}
        k1_new = feval(field, t(i)+h, x(i,:).'+h*k1_old); % k1^{(n+1)}
    end
    x(i+1,:) = x(i,:).' + h*k1_new;
end