function [t,x] = crank_nicholson(field, t0, T, x0, h, tol)
%CRANK_NICHOLSON Implements the Crank-Nicholson one-step ODE solver
%
% Parameters:
%  field  -- Right hand side function of ODE system: x'=f(t,x)
%  t0     -- Initial time
%  T      -- End time (T > t0)
%  x0     -- Initial value
%  h      -- Size of time step (h <= T-t0)
%  tol    -- Tolerance to use for solving the fixed point iteration (default=0.01)
%
% Outputs:
%  t  -- [t0; t-0+h, t0+2*h; ...; t0+i*h; ...]
%  x  -- Matrix containing numerical solution, with each row the value of x
%        at each time step

arguments
    field; t0; T; x0; h;
    tol = 0.01;
end

n = ceil((T-t0)/h);
t = t0+h*(0:n).';
x = ones(n+1,length(x0));
x(1,:) = x0;

for i=1:n
    k1 = feval(field, t(i), x(i,:).');
    k2_old = k1; % k2_old = k2^{(0)}
    k2_new = feval(field, t(i)+h, x(i,:).'+h*(k1+k2_old)/2); % k2_new = k2^{(1)}
    while norm(k2_old-k2_new, 'inf') >= tol
        k2_old = k2_new; % k2^{(n)}
        k2_new = feval(field, t(i)+h, x(i,:).'+h*(k1+k2_old)/2); % k2^{(n+1)}
    end
    x(i+1,:) = x(i,:).' + h*(k1+k2_new)/2;
end
