function [pts, weights] = quadrature_tri(p)
%QUADRATURE_TRI Returns the weights and quadrature points for a triangle
%
% Arguments:
%   p  - Desired order of quadrature
%
% Returns:
%   pts     - Nx2 matrix (N is number of quadrature points) containing,
%             per row, the X & Y coordinate of the quadrature point in the
%             reference triangle (-1,-1)-(1,-1)-(-1,1);
%   weights - Vector of length N containing the quadrature weight
%             corresponding to each quadrarure point.
p = round(p);
if p < 0
    p = 1;
end
switch p
    case 1
        A = [1 1/3 1/3 1/3];
    case 2
        A = rotatecoords(1/3, 2/3, 1/6, 1/6);
    case 3
        A = [-9/16 1/3 1/3 1/3; rotatecoords(25/48, 3/5, 1/5, 1/5)];
    case 4
        A = [rotatecoords(0.223381589678011, 0.108103018168070, 0.445948490915965, 0.445948490915965);
            rotatecoords(0.109951743655322, 0.816847572980459, 0.091576213509771, 0.091576213509771)];
    case 5
        A = [9/40 1/3 1/3 1/3;
            rotatecoords(0.132394152788506, 0.059715871789770, 0.4701420641051, 0.4701420641051);
            rotatecoords(0.125939180544827, 0.797426985353087, 0.101286507323456, 0.101286507323456)];
    case 6
        A = [rotatecoords(0.116786275726379, 0.501426509658179, 0.249286745170910, 0.249286745170910);
            rotatecoords(0.050844906370207, 0.873821971016996, 0.063089014491502, 0.063089014491502);
            permutecoords(0.082851075618374, 0.053145049844817, 0.310352451033784, 0.636502499121399)];
    case 7
        A = [-0.149570044467682 1/3 1/3 1/3;
            rotatecoords(0.175615257433208, 0.479308067841920, 0.260345966079040, 0.260345966079040);
            rotatecoords(0.053347235608838, 0.869739794195568, 0.065130102902216, 0.065130102902216);
            permutecoords(0.077113760890257, 0.048690315425316, 0.312865496004874, 0.638444188569810)];
    case 8
        A = [0.144315607677787 1/3 1/3 1/3;
            rotatecoords(0.095091634267285, 0.081414823414554, 0.459292588292723, 0.459292588292723);
            rotatecoords(0.103217370534718, 0.658861384496480, 0.170569307751760, 0.170569307751760);
            rotatecoords(0.032458497623198, 0.898905543365938, 0.050547228317031, 0.050547228317031);
            permutecoords(0.027230314174435, 0.008394777409958, 0.263112829634638, 0.728492392955404)];
    case 9
        A = [0.097135796282799 1/3 1/3 1/3;
            rotatecoords(0.031334700227139, 0.020634961602525, 0.489682519198738, 0.489682519198738);
            rotatecoords(0.077827541004774, 0.125820817014127, 0.437089591492937, 0.437089591492937);
            rotatecoords(0.079647738927210, 0.623592928761935, 0.188203535619033, 0.188203535619033);
            rotatecoords(0.025577675658698, 0.910540973211095, 0.044729513394453, 0.044729513394453);
            permutecoords(0.043283539377289, 0.036838412054736, 0.221962989160766, 0.741198598784498)];
    case 10
        A = [0.090817990382754 1/3 1/3 1/3;
            rotatecoords(0.036725957756467, 0.028844733232685, 0.485577633383657, 0.485577633383657);
            rotatecoords(0.045321059435528, 0.781036849029926, 0.109481575485037, 0.109481575485037);
            permutecoords(0.072757916845420, 0.141707219414880, 0.307939838764121, 0.550352941820999);
            permutecoords(0.028327242531057, 0.025003534762686, 0.246672560639903, 0.728323904597411);
            permutecoords(0.009421666963733, 0.009540815400299, 0.066803251012200, 0.923655933587500)];
    case 11
        A = [rotatecoords(0.000927006328961, -0.069222096541517, 0.534611048270758, 0.534611048270758);
            rotatecoords(0.077149534914813, 0.202061394068290, 0.398969302965855, 0.398969302965855);
            rotatecoords(0.059322977380774, 0.593380199137435, 0.203309900431282, 0.203309900431282);
            rotatecoords(0.036184540503418, 0.761298175434837, 0.119350912282581, 0.119350912282581);
            rotatecoords(0.013659731002678, 0.935270103777448, 0.032364948111276, 0.032364948111276);
            permutecoords(0.052337111962204, 0.050178138310495, 0.356620648261293, 0.593201213428213);
            permutecoords(0.020707659639141, 0.021022016536166, 0.171488980304042, 0.807489003159792)];
    case 12
        A = [rotatecoords(0.025731066440455, 0.023565220452390, 0.488217389773805, 0.488217389773805);
            rotatecoords(0.043692544538038, 0.120551215411079, 0.439724392294460, 0.439724392294460);
            rotatecoords(0.062858224217885, 0.457579229975768, 0.271210385012116, 0.271210385012116);
            rotatecoords(0.034796112930709, 0.744847708916828, 0.127576145541586, 0.127576145541586);
            rotatecoords(0.006166261051559, 0.957365299093579, 0.021317350453210, 0.021317350453210);
            permutecoords(0.040371557766381, 0.115343494534698, 0.275713269685514, 0.608943235779788);
            permutecoords(0.022356773202303, 0.022838332222257, 0.281325580989940, 0.695836086787803);
            permutecoords(0.017316231108659, 0.025734050548330, 0.116251915907597, 0.858014033544073)];
    case 13
        A = [0.052520923400802 1/3 1/3 1/3;
            rotatecoords(0.011280145209330, 0.009903630120591, 0.495048184939705, 0.495048184939705);
            rotatecoords(0.031423518362454, 0.062566729780852, 0.468716635109574, 0.468716635109574);
            rotatecoords(0.047072502504194, 0.170957326397447, 0.414521336801277, 0.414521336801277);
            rotatecoords(0.047363586536355, 0.541200855914337, 0.229399572042831, 0.229399572042831);
            rotatecoords(0.031167529045794, 0.771151009607340, 0.114424495196330, 0.114424495196330);
            rotatecoords(0.007975771465074, 0.950377217273082, 0.024811391363459, 0.024811391363459);
            permutecoords(0.036848402728732, 0.094853828379579, 0.268794997058761, 0.636351174561660);
            permutecoords(0.017401463303822, 0.018100773278807, 0.291730066734288, 0.690169159986905);
            permutecoords(0.015521786839045, 0.022233076674090, 0.126357385491669, 0.851409537834241)];
    case 14
        A = [rotatecoords(0.021883581369429, 0.022072179275643, 0.488963910362179, 0.488963910362179);
            rotatecoords(0.032788353544125, 0.164710561319092, 0.417644719340454, 0.417644719340454);
            rotatecoords(0.051774104507292, 0.453044943382323, 0.273477528308839, 0.273477528308839);
            rotatecoords(0.042162588736993, 0.645588935174913, 0.177205532412543, 0.177205532412543);
            rotatecoords(0.014433699669777, 0.876400233818255, 0.061799883090873, 0.061799883090873);
            rotatecoords(0.004923403602400, 0.961218077502598, 0.019390961248701, 0.019390961248701);
            permutecoords(0.024665753212564, 0.057124757403648, 0.172266687821356, 0.770608554774996);
            permutecoords(0.038571510787061, 0.092916249356972, 0.336861459796345, 0.570222290846683);
            permutecoords(0.014436308113534, 0.014646950055654, 0.298372882136258, 0.686980167808088);
            permutecoords(0.005010228838501, 0.001268330932872, 0.118974497696957, 0.879757171370171)];
    case 15
        A = [rotatecoords(0.001916875642849, -0.013945833716486, 0.506972916858243, 0.506972916858243);
            rotatecoords(0.044249027271145, 0.137187291433955, 0.431406354283023, 0.431406354283023);
            rotatecoords(0.051186548718852, 0.444612710305711, 0.277693644847144, 0.277693644847144);
            rotatecoords(0.023687735870688, 0.747070217917492, 0.126464891041254, 0.126464891041254);
            rotatecoords(0.013289775690021, 0.858383228050628, 0.070808385974686, 0.070808385974686);
            rotatecoords(0.004748916608192, 0.962069659517853, 0.018965170241073, 0.018965170241073);
            permutecoords(0.038550072599593, 0.133734161966621, 0.261311371140087, 0.604954466893291);
            permutecoords(0.027215814320624, 0.036366677396917, 0.388046767090269, 0.575586555512814);
            permutecoords(0.002182077366797, -0.010174883126571, 0.285712220049916, 0.724462663076655);
            permutecoords(0.021505319847731, 0.036843869875878, 0.215599664072284, 0.747556466051838);
            permutecoords(0.007673942631049, 0.012459809331199, 0.103575616576386, 0.883964574092416)];
    case 16
        A = [0.046875697427642 1/3 1/3 1/3;
            rotatecoords(0.006405878578585, 0.005238916103123, 0.497380541948438, 0.497380541948438);
            rotatecoords(0.041710296739387, 0.173061122901295, 0.413469438549352, 0.413469438549352);
            rotatecoords(0.026891484250064, 0.059082801866017, 0.470458599066991, 0.470458599066991);
            rotatecoords(0.042132522761650, 0.518892500060958, 0.240553749969521, 0.240553749969521);
            rotatecoords(0.030000266842773, 0.704068411554854, 0.147965794222573, 0.147965794222573);
            rotatecoords(0.014200098925024, 0.849069624685052, 0.075465187657474, 0.075465187657474);
            rotatecoords(0.003582462351273, 0.966807194753950, 0.016596402623025, 0.016596402623025);
            permutecoords(0.032773147460627, 0.103575692245252, 0.296555596579887, 0.599868711174861);
            permutecoords(0.015298306248441, 0.020083411655416, 0.337723063403079, 0.642193524941505);
            permutecoords(0.002386244192839, -0.004341002614139, 0.204748281642812, 0.799592720971327);
            permutecoords(0.019084792755899, 0.041941786468010, 0.189358492130623, 0.768699721401368);
            permutecoords(0.006850054546542, 0.014317320230681, 0.085283615682657, 0.900399064086661)];
    case 17
        A = [0.033437199290803 1/3 1/3 1/3;
            rotatecoords(0.005093415440507, 0.005658918886452, 0.497170540556774, 0.497170540556774);
            rotatecoords(0.014670864527638, 0.035647354750751, 0.482176322624625, 0.482176322624625);
            rotatecoords(0.024350878353672, 0.099520061958437, 0.450239969020782, 0.450239969020782);
            rotatecoords(0.031107550868969, 0.199467521245206, 0.400266239377397, 0.400266239377397);
            rotatecoords(0.031257111218620, 0.495717464058095, 0.252141267970953, 0.252141267970953);
            rotatecoords(0.024815654339665, 0.675905990683077, 0.162047004658461, 0.162047004658461);
            rotatecoords(0.014056073070557, 0.848248235478508, 0.075875882260746, 0.075875882260746);
            rotatecoords(0.003194676173779, 0.968690546064356, 0.015654726967822, 0.015654726967822);
            permutecoords(0.008119655318993, 0.010186928826919, 0.334319867363658, 0.655493203809423);
            permutecoords(0.026805742283163, 0.135440871671036, 0.292221537796944, 0.572337590532020);
            permutecoords(0.018459993210822, 0.054423924290583, 0.319574885423190, 0.626001190286228);
            permutecoords(0.008476868534328, 0.012868560833637, 0.190704224192292, 0.796427214974071);
            permutecoords(0.018292796770025, 0.067165782413524, 0.180483211648746, 0.752351005937729);
            permutecoords(0.006665632004165, 0.014663182224828, 0.080711313679564, 0.904625504095608)];
    case 18
        A = [0.030809939937647 1/3 1/3 1/3;
            rotatecoords(0.009072436679404, 0.013310382738157, 0.493344808630921, 0.493344808630921);
            rotatecoords(0.018761316939594, 0.061578811516086, 0.469210594241957, 0.469210594241957);
            rotatecoords(0.019441097985477, 0.127437208225989, 0.436261395887006, 0.436281395887006);
            rotatecoords(0.027753948610810, 0.210307658653168, 0.394846170673416, 0.394846170673416);
            rotatecoords(0.032256225351457, 0.500410862393686, 0.249794568803157, 0.249794568803157);
            rotatecoords(0.025074032616922, 0.677135612512315, 0.161432193743843, 0.161432193743843);
            rotatecoords(0.015271927971832, 0.846803545029257, 0.076598227485371, 0.076598227485371);
            rotatecoords(0.006793922022963, 0.951495121293100, 0.024252439353450, 0.024252439353450);
            rotatecoords(-0.002223098729920, 0.913707265566071, 0.043146367216965, 0.043146367216965);
            permutecoords(0.006331914076406, 0.008430536202420, 0.358911494940944, 0.632657968856636);
            permutecoords(0.027257538049138, 0.131186551737188, 0.294402476751957, 0.574410971510855);
            permutecoords(0.017676785649465, 0.050203151565675, 0.325017801641814, 0.624779046792512);
            permutecoords(0.018379484638070, 0.066329263810916, 0.184737559666046, 0.748933176523037);
            permutecoords(0.008104732808192, 0.011996194566236, 0.218796800013321, 0.769207005420443);
            permutecoords(0.007634129070725, 0.014858100590125, 0.101179597136408, 0.883962302273467);
            permutecoords(0.000046187660794, -0.035222015287949, 0.020874755282586, 1.014347260005363)];
    case 19
        A = [0.032906331388919 1/3 1/3 1/3;
            rotatecoords(0.010330731891272, 0.020780025853987, 0.489609987073006, 0.489609987073006);
            rotatecoords(0.022387247263016, 0.090926214604215, 0.454536892697893, 0.454536892697893);
            rotatecoords(0.030266125869468, 0.197166638701138, 0.401416680649431, 0.401416680649431);
            rotatecoords(0.030490967802198, 0.488896691193805, 0.255551654403098, 0.255551654403098);
            rotatecoords(0.024159212741641, 0.645844115695741, 0.177077942152130, 0.177077942152130);
            rotatecoords(0.016050803586801, 0.779877893544096, 0.110061053227952, 0.110061053227952);
            rotatecoords(0.008084580261784, 0.888942751496321, 0.055528624251840, 0.055528624251840);
            rotatecoords(0.002079362027485, 0.974756272445543, 0.012621863777229, 0.012621863777229);
            permutecoords(0.003884876904981, 0.003611417848412, 0.395754787356943, 0.600633794794645);
            permutecoords(0.025574160612022, 0.134466754530780, 0.307929983880436, 0.557603261588784);
            permutecoords(0.008880903573338, 0.014446025776115, 0.264566948406520, 0.720987025817365);
            permutecoords(0.016124546761731, 0.046933578838178, 0.358539352205951, 0.594527068955871);
            permutecoords(0.002861120350567, 0.002491941817491, 0.157807405968595, 0.839331473680839);
            permutecoords(0.018242840118951, 0.223861424097916, 0.075050596975911, 0.701087978926173);
            permutecoords(0.010258563736199, 0.034647074816760, 0.142421601113383, 0.822931324069857);
            permutecoords(0.003799928855302, 0.010161119296278, 0.065494628082938, 0.924344252620784)];
    otherwise % p=20 or greater
        A = [0.033057055541624 1/3 1/3 1/3;
            rotatecoords(0.000867019185663, -0.001900928704400, 0.500950464352200, 0.500950464352200);
            rotatecoords(0.011660052716448, 0.023574084130543, 0.488212957934729, 0.488212957934729);
            rotatecoords(0.022876936356421, 0.089726636099435, 0.455136681950283, 0.455136681950283);
            rotatecoords(0.030448982673938, 0.196007481363421, 0.401996259318289, 0.401996259318289);
            rotatecoords(0.030624891725355, 0.488214180481157, 0.255892909759421, 0.255892909759421);
            rotatecoords(0.024368057676800, 0.647023488009788, 0.176488255995106, 0.176488255995106);
            rotatecoords(0.015997432032024, 0.791658289326483, 0.104170855336758, 0.104170855336758);
            rotatecoords(0.007698301815602, 0.893862072318140, 0.053068963840930, 0.053068963840930);
            rotatecoords(-0.000632060497488, 0.916762569607942, 0.041618715196029, 0.041618715196029);
            rotatecoords(0.001751134301193, 0.976836157186356, 0.011581921406822, 0.011581921406822);
            rotatecoords(0.016465839189576, 0.048741583664839, 0.344855770229001, 0.606402646106160);
            permutecoords(0.004839033540485, 0.006314115948605, 0.377843269594854, 0.615842614456541);
            permutecoords(0.025804906534650, 0.134316520547348, 0.306635479062357, 0.559048000390295);
            permutecoords(0.008471091054441, 0.013973893962392, 0.249419362774742, 0.736606743262866);
            permutecoords(0.018354914106280, 0.075549132909764, 0.212775724802802, 0.711675142287434);
            permutecoords(0.000704404677908, -0.008368153208227, 0.146965436053239, 0.861402717154987);
            permutecoords(0.010112684927462, 0.026686063258714, 0.137726978828923, 0.835586957912363);
            permutecoords(0.003573909385950, 0.010547719294141, 0.059696109149007, 0.929756171556853)];
end

weights = 2*A(:,1);
pts = A(:,2:4)*[-1 -1; 1 -1; -1 1];

end

function A = rotatecoords(w, x1, x2, x3)
%ROTATECOORDS Returns a matrix with three rows containing the weight and the
%three barycentric points coords rotated
A = [w x1 x2 x3; w x2 x3 x1; w x3 x1 x2];
end

function A = permutecoords(w, x1, x2, x3)
%PERMUTECOORDS Returns a matrix containing every permuation of the barycentric
%coordinates
A = [w x1 x2 x3; w x1 x3 x2; w x2 x1 x3; w x2 x3 x1; w x3 x1 x2; w x3 x2 x1];
end