function [pts, weights] = quadrature_rect(p)
%QUADRATURE_RECT Quadrature points/weights for reference square [-1,1]^2
%
% Arguments:
%    p - Number of desired quadrature points in one dimension
%
% Returns:
%   pts     - Nx2 matrix (N=p^2 is number of quadrature points) containing,
%             per row, the X & Y coordinate of the quadrature point in the
%             reference square [-1,1]^2
%   weights - Vector of length N containing the quadrature weight
%             corresponding to each quadrarure point.

% Get 1D quadrature points and then simply tensor/cartesian product.
[x,w] = quadrature_1d(p);
[X,Y] = meshgrid(x,x);
weights = reshape(w*w', [], 1);
pts = [reshape(X, [], 1), reshape(Y, [], 1)];

end

function [x, w] = quadrature_1d(n)
%QUADRATURE_1D Quadrature points/weights for reference interval [-1,1]
% Arguments:
%    n - Number of desired quadrature points
%
% Returns two vectors of length n containing the points and weights
% respectively.

% This method just finds the roots of the Legendre polynomial, using
% "sensible" well-known initial guesses to ensure all roots are found
x = zeros(n,1);
w = zeros(n,1);

eps = 3e-15;
m = floor((n+1)/2);
for i=1:m
    z = cos(pi*(i-0.25)/(n+0.5));
    z1 = z+1;
    while (abs(z-z1) > eps)
        p1 = 1;
        p2 = 0;
        for j=1:n
            p3 = p2;
            p2 = p1;
            p1 = ((2*j-1)*z*p2-(j-1)*p3)/j;
        end
        pp = n*(z*p1-p2)/(z^2-1);
        z1 = z;
        z = z1-p1/pp;
    end
    x(i) = -z;
    x(n+1-i) = z;
    w(i) = 2/((1-z^2)*pp^2);
    w(n+1-i) = w(i);
end

end