function fig = plot_soln(basis,polydeg,mesh,u)
%PLOT_SOLN Plots the solution obtained by FEM
%
% Note, if basis and polydeg are not compatible with the mesh and solution
% an error will be thrown.
%
% Arguments:
%    basis   - The basis which was used by FEM, see fem.m.
%    polydeg - The polynomial degree of the basis used by FEM
%    mesh    - Mesh data returned by FEM
%    u       - Solution vector returned by FEM

nv = 3;
ndofs = 0;
switch basis
    case {'tri', 'triangle', 'simplex'}
        if polydeg < 1
            error('Only p>=1 supported for Lagrange simplex element')
        end
        [pts,tri,bdry] = tri_surface(polydeg);
        basis = basis_lagrange_tri(polydeg,pts(:,1),pts(:,2));
        ndofs = (polydeg*(sqrt(size(mesh.vertices,1))-1)+1)^2;
    case {'quad','quadrilateral','rect','rectangle'}
        if polydeg < 1
            error('Only p>=1 supported for Lagrange quadrilateral element')
        end
        [pts,tri,bdry] = quad_surface(polydeg);
        basis = basis_lagrange_quad(polydeg,pts(:,1),pts(:,2));
        nv = 4;
        ndofs = (polydeg*(sqrt(size(mesh.vertices,1))-1)+1)^2;
    case {'reduced_tri', 'reduced_triangle', 'reduced_simplex'}
        if polydeg ~= 3
            error('Only p=3 supported for reduced Lagrange simplex element')
        end
        [pts,tri,bdry] = tri_surface(polydeg);
        basis = basis_reduced_tri(polydeg,pts(:,1),pts(:,2));
        ndofs = (polydeg*(sqrt(size(mesh.vertices,1))-1)+1)^2-size(mesh.elements,1);
    case {'reduced_quad','reduced_quadrilateral','reduced_rect','reduced_rectangle'}
        if polydeg < 2 || polydeg > 3
            error('Only p=2,3 supported for reduced Lagrange quadrilateral element')
        end
        [pts,tri,bdry] = quad_surface(polydeg);
        basis = basis_reduced_quad(polydeg,pts(:,1),pts(:,2));
        nv = 4;
        ndofs = (polydeg*(sqrt(size(mesh.vertices,1))-1)+1)^2 ...
            -size(mesh.elements,1)*(polydeg-1)^2;
    case {'cr', 'crouzeix_raviart'}
        if polydeg ~= 1
            error('Only p=1 supported for Crouzeix-Raviart element')
        end
        [pts,tri,bdry] = tri_surface(1);
        basis = basis_crouzeix_raviart(pts(:,1),pts(:,2));
        nverts = sqrt(size(mesh.vertices,1));
        ndofs = (nverts-1)*(3*nverts-1);
    case {'rotated_quad','rotated_quadrilateral','rotated_rect','rotated_rectangle','rotated_bilinear'}
        if polydeg ~= 1
            error('Only p=1 supported for rotated bilinear element')
        end
        [pts,tri,bdry] = quad_surface(2);
        basis = basis_rotated_bilinear(pts(:,1),pts(:,2));
        nv = 4;
        nverts = sqrt(size(mesh.vertices,1));
        ndofs = 2*(nverts-1)*nverts;
    otherwise
        error('Unknown basis type');
end

if (size(basis,2) ~= size(mesh.dof_mapping, 2)) || ...
        (nv ~= size(mesh.elements, 2)) || ...
        (ndofs ~= length(u))
    error('Mesh and solution data not for the specified basis');
end

n = size(pts,1);
nt = size(tri,1);
X = zeros(n*size(mesh.elements,1),2);
Z = zeros(n*size(mesh.elements,1),1);
T = zeros(nt*size(mesh.elements,1),3);
P = zeros(size(mesh.elements,1),length(bdry));

for ele=1:size(mesh.elements,1)
    vertices = mesh.vertices(mesh.elements(ele,:), :)';
    if size(vertices,2) == 3
        % Triangles
        [B, c] = tri_mapping(vertices);
    else
        % Quads
        [B, c] = quad_mapping(vertices);
    end
    B = B';
    c = repmat(c', size(pts,1), 1);
    X((n*(ele-1)+1):n*ele,:) = pts*B+c;
    T((nt*(ele-1)+1):nt*ele,:) = tri + n*(ele-1);
    Z((n*(ele-1)+1):n*ele) = basis*u(mesh.dof_mapping(ele,:));
    P(ele,:) = bdry + n*(ele-1);
end

fig = figure;
trisurf(T, X(:,1), X(:,2), Z, 'EdgeColor','none','FaceColor','interp');
hold on;
patch('Faces', P, 'Vertices', [X Z], ...
    'EdgeColor', 'k', 'FaceColor', 'none', 'LineWidth', 1);
end

function [pts,tri,bdry] = tri_surface(polydeg)
%TRI_SURFACE Constructs a triangulation for the reference triangle which
% will allow a sensible interpolation of the solution to be displayed using
% trisurf or patch
%
% Arguments:
%   polydeg - Polynomial degree of basis functions to interpolate
% 
% Returns:
%   pts  - Nx2 matrix of points to interpolate the result at in the
%          reference triangle (pass pts(:,1) and pts(:,2) to the first and
%          second arguments of trisurf, respectively);
%   tri  - Mx3 matrix of indices to triangulate the points (pass to first 
%          argument of trisurf)
%   bdry - Vector of indices to denote the boundary points of the element
%          (can be used to construct a polygon boundary line for the ele)
if polydeg == 1
    pts = [-1 -1; 1 -1; -1 1];
    tri = [1 2 3];
    bdry = tri;
else
    n = 5*polydeg;
    x = linspace(-1, 1, n+1);
    npts = (n+2)*(n+1)/2;
    pts = zeros(npts,2);
    tri = zeros(n*n, 3);
    startidx = 1;
    starttri = 1;
    bdry = zeros(1,3*n);
    bdry(1:n-1) = 2:n;
    for i=1:n
        bdry(3*n+1-i) = startidx;
        bdry(n-1+i) = startidx+n+1-i;
        
        pts(startidx:(startidx+n+1-i),1) = x(1:n+2-i);
        pts(startidx:(startidx+n+1-i),2) = x(i);
        tri(starttri:2:(starttri+2*(n+1-i)-1),1) = startidx:(startidx+n-i);
        tri(starttri:2:(starttri+2*(n+1-i)-1),2) = (startidx+1):(startidx+1+n-i);
        tri(starttri:2:(starttri+2*(n+1-i)-1),3) = (startidx+n+2-i):(startidx+n+2-i+n-i);
        if i < n
            tri((starttri+1):2:(starttri+2*(n+1-i)-1),1) = (startidx+1):(startidx+1+n-i);
            tri((starttri+1):2:(starttri+2*(n+1-i)-1),2) = (startidx+n+3-i):(startidx+n+3-i+n-i);
            tri((starttri+1):2:(starttri+2*(n+1-i)-1),3) = (startidx+n+2-i):(startidx+n+2-i+n-i);
        end
        startidx = startidx+n+2-i;
        starttri = starttri+2*(n+1-i);
    end
    bdry(2*n) = startidx;
    pts(startidx,1) = x(1);
    pts(startidx,2) = x(n+1);
end

end

function [pts,tri,bdry] = quad_surface(polydeg)
%QUAD_SURFACE Constructs a triangulation for the reference square which
% will allow a sensible interpolation of the solution to be displayed using
% trisurf or patch
%
% Arguments:
%   polydeg - Polynomial degree of basis functions to interpolate
% 
% Returns:
%   pts  - Nx2 matrix of points to interpolate the result at in the
%          reference square (pass pts(:,1) and pts(:,2) to the first and
%          second arguments of trisurf, respectively);
%   tri  - Mx4 matrix of indices to triangulate the points (pass to first 
%          argument of trisurf)
%   bdry - Vector of indices to denote the boundary points of the element
%          (can be used to construct a polygon boundary line for the ele)

n = 5*(polydeg+1);
x = linspace(-1, 1, n+1);
[X, Y] = meshgrid(x, x);
I = reshape(1:(n+1)^2, n+1, n+1);
bdry = [I(:,1)' I(n+1,2:n) I(n+1:-1:1,n+1)' I(1,n:-1:2)]; 
tri = [reshape(I(1:n, 1:n), [], 1) reshape(I(1:n, 2:n+1), [], 1) reshape(I(2:n+1, 1:n), [], 1);
    reshape(I(2:n+1, 2:n+1), [], 1) reshape(I(2:n+1, 1:n), [], 1) reshape(I(1:n, 2:n+1), [], 1)];
pts = [reshape(X, [], 1) reshape(Y, [], 1)];

end

function [B, b] = quad_mapping(vertices)
%QUAD_MAPPING Returns affine map from [-1,1]^2 to specified element
% Assumes we have parallelogram - ensures affine map
%
% Arguments:
%   vertices - 4x2 matrix containing the coordinates of the element in
%              counter-clockwise order
%
% Returns:
%   [B, b] - Matrix and vector to perform mapping from reference: x = B*x+b

B = [vertices(:,2)-vertices(:,1) vertices(:,4)-vertices(:,1)]/2;
b = (vertices(:,1)+vertices(:,2)+vertices(:,3)+vertices(:,4))/4;
end

function [B, b] = tri_mapping(vertices)
%QUAD_MAPPING Returns affine map from (-1 -1)-(1,-1)-(-1,1) triangle
%
% Arguments:
%   vertices - 3x2 matrix containing the coordinates of the element in
%              counter-clockwise order
%
% Returns:
%   [B, b] - Matrix and vector to perform mapping from reference: x = B*x+b

B = [vertices(:,2)-vertices(:,1)  vertices(:,3)-vertices(:,1)]/2;
b = (vertices(:,2)+vertices(:,3))/2;
end
