function [DX, DY] = grad_basis_lagrange_tri(p, X, Y)
%GRAD_BASIS_LAGRANGE_TRI Returns gradient of total-order Lagrange basis 
%  evaluated at points in the reference triangle (-1,-1)-(1,-1)-(-1,1)
%
% Arguments:
%     p - Polynomial degree of Lagrange polynomial
%     X - Vector of x-coordinates of points to evaluate
%     Y - Vector of y-coordinates of points to evaluate
%
% Returns:
%     DX - Nx((p+1)(p+2)/2) matrix, where N is the number of points,
%          containing per row the values of the derivatives of the basis
%          w.r.t. X evaluated at the corresponding point. See 
%          basis_lagrange_tri for details on each basis point.
%     DY - Nx((p+1)(p+2)/2) matrix containing derivatives of basis w.r.t Y
if p < 1
    error('Lagrange basis on triangle must be at least polynomial degree 1');
end

lambda = zeros(length(X),3);
% Barycentric coordinates
lambda(:,1) = -(X+Y)/2;
lambda(:,2) = (1+X)/2;
lambda(:,3) = (1+Y)/2;
% Derivatives of barycentric
lambdaX = [-1/2; 1/2; 0];
lambdaY = [-1/2; 0; 1/2];

DX = zeros(length(X), (p+2)*(p+1)/2);
DY = zeros(length(X), (p+2)*(p+1)/2);

for idx=1:3
    % Vertex basis
    D = zeros(size(X));
    factor = 1;
    for i=0:(p-1)
        Z = ones(size(X));
        for j=0:(p-1)
            if j ~= i
                Z = Z .* (lambda(:,idx) - j/p);
            end
        end
        factor = factor * (1-i/p);
        D = D+Z;
    end
    DX(:,idx) = lambdaX(idx)*D/factor;
    DY(:,idx) = lambdaY(idx)*D/factor;
end
index = 4;
if p > 1
    % Edge basis
    for v1=1:3
        v2 = v1 + 1;
        if v2 > 3
            v2 = 1;
        end
        for idx=1:(p-1)
            for i=[0:(idx-1) (idx+1):p]
                Z = ones(size(X));
                factor = 1;
                for k=0:(idx-1)
                    if k ~= i
                        Z = Z .* (lambda(:,v2) - k/p);
                    end
                    factor = factor * (idx-k)/p;
                end
                for k=(idx+1):p
                    if k ~= i
                        Z = Z .* (lambda(:,v1) - (p-k)/p);
                    end
                    factor = factor * (k-idx)/p;
                end
                if (i < idx)
                    DX(:,index) = DX(:,index) + Z*lambdaX(v2)/factor;
                    DY(:,index) = DY(:,index) + Z*lambdaY(v2)/factor;
                else
                    DX(:,index) = DX(:,index) + Z*lambdaX(v1)/factor;
                    DY(:,index) = DY(:,index) + Z*lambdaY(v1)/factor;
                end
            end
            index = index+1;
        end
    end
end
if p > 2
    % Interior basis
    for j = 1:p-2
        for i = 1:p-1-j
            for m=0:(p-1)
                factor = 1;
                Z = ones(size(X));
                for k=0:i-1
                    if k ~= m
                        Z = Z .* (lambda(:,2)-k/p);
                    end
                    factor = factor * (i-k)/p;
                end
                for k=0:j-1
                    if k ~= m-i
                        Z = Z .* (lambda(:,3)-k/p);
                    end
                    factor = factor * (j-k)/p;
                end
                for k=(i+j+1):p
                    if k ~= m+1
                        Z = Z .* (lambda(:,1)-(p-k)/p);
                    end
                    factor = factor * (k-i-j)/p;
                end
                if (m < i)
                    DX(:,index) = DX(:,index) + Z*lambdaX(2)/factor;
                    DY(:,index) = DY(:,index) + Z*lambdaY(2)/factor;
                elseif (m < i+j)
                    DX(:,index) = DX(:,index) + Z*lambdaX(3)/factor;
                    DY(:,index) = DY(:,index) + Z*lambdaY(3)/factor;
                else
                    DX(:,index) = DX(:,index) + Z*lambdaX(1)/factor;
                    DY(:,index) = DY(:,index) + Z*lambdaY(1)/factor;
                end
            end
            index = index+1;
        end
    end
end

end