function [DX, DY] = grad_basis_lagrange_quad(p, X, Y)
%GRAD_BASIS_LAGRANGE_QUAD Returns gradient of tensor Lagrange basis 
%  evaluated at points in the reference square [-1,1]^2
%
% Arguments:
%     p - Polynomial degree of Lagrange polynomial
%     X - Vector of x-coordinates of points to evaluate
%     Y - Vector of y-coordinates of points to evaluate
%
% Returns:
%     DX - Nx((p+1)^2) matrix, where N is the number of points, containing
%          per row the values of the derivatives of the basis w.r.t. X
%          evaluated at the corresponding point. See basis_lagrange_quad
%          for details on each basis point.
%     DY - Nx((p+1)^2) matrix containing the derivatives of basis w.r.t Y

if p < 1
    error('Lagrange basis on rectangles must be at least polynomial degree 1');
end

ZX = basis_lagrange_interval(p, X);
ZY = basis_lagrange_interval(p, Y);
ZDX = deriv_basis_lagrange_interval(p, X);
ZDY = deriv_basis_lagrange_interval(p, Y);

DX = zeros(length(X), (p+1)^2);
DX(:,1) = ZDX(:,1).*ZY(:,1);
DX(:,2) = ZDX(:,p+1).*ZY(:,1);
DX(:,3) = ZDX(:,p+1).*ZY(:,p+1);
DX(:,4) = ZDX(:,1).*ZY(:,p+1);

DY = zeros(length(X), (p+1)^2);
DY(:,1) = ZX(:,1).*ZDY(:,1);
DY(:,2) = ZX(:,p+1).*ZDY(:,1);
DY(:,3) = ZX(:,p+1).*ZDY(:,p+1);
DY(:,4) = ZX(:,1).*ZDY(:,p+1);

if p > 1
    DX(:,5:(3+p)) = ZY(:,1).*ZDX(:,2:p);
    DY(:,5:(3+p)) = ZDY(:,1).*ZX(:,2:p);
    DX(:,(4+p):(2+2*p)) = ZDX(:,p+1).*ZY(:,2:p);
    DY(:,(4+p):(2+2*p)) = ZX(:,p+1).*ZDY(:,2:p);
    DX(:,(3+2*p):(3*p+1)) = ZY(:,p+1).*ZDX(:,p:-1:2);
    DY(:,(3+2*p):(3*p+1)) = ZDY(:,p+1).*ZX(:,p:-1:2);
    DX(:,(2+3*p):(4*p)) = ZDX(:,1).*ZY(:,p:-1:2);
    DY(:,(2+3*p):(4*p)) = ZX(:,1).*ZDY(:,p:-1:2);
    for i=2:p
        DX(:,(5+(i+2)*(p-1)):(4+(i+3)*(p-1))) = ZDX(:,2:p).*ZY(:,i);
        DY(:,(5+(i+2)*(p-1)):(4+(i+3)*(p-1))) = ZX(:,2:p).*ZDY(:,i);
    end
end

end