function [l2, h1] = fem_convergence(basis,p,max_itn)
%FEM_CONVERGENCE Returns FEM for specified basis at several element sizes
% plotting errors and estimating the convergence rate of the method
%
% Runs FEM with n=2^j, j=1,...,max_itns
%
% Arguments:
%     basis   - Name of basis to use for FEM. See fem for list of
%               supported basis names
%     p       - Polynomial degree of basis
%     max_itn - Last "iteration" of FEM (n=2^max_itns). Optional argument,
%               if not specified max_itn = 6 is used.
%
% Returns:
%     l2 - Estimate of rate of convergence of error in L^2 norm
%     h1 - Estimate of rate of convergence of error in H^1 seminorm

% Check if optional argument specified
if ~exist('max_itn','var')
    max_itn = 6;
end

h = zeros(1,1);
l2errs = zeros(1,1);
h1errs = zeros(1,1);

% Display initial empty plot of convergence rates
l2p = loglog(h, l2errs, '-rx', 'DisplayName', 'L^2 Error');
hold on;
h1p = loglog(h, h1errs, '-bo', 'DisplayName', 'H^1 Error');
legend('Location','SouthEast');
xlabel('h');
ylabel('Error');
ylim([1e-10, 10]);
xlim([1/(2.^(max_itn+1)) 1]);

% Loop through different number of elements
for n=1:max_itn
    fprintf('Solving for n=%d\n', 2^n);
    % Solve
    [~,~,~,l2err,h1err] = fem(2^n,basis,p);
    
    % Update plot
    l2errs(n) = l2err;
    h1errs(n) = h1err;
    h(n) = 1./(2.^n);
    set(l2p, 'XData', h, 'YData', l2errs);
    set(h1p, 'XData', h, 'YData', h1errs);
    pause(.5);
end

% Compute convergence rates in L^2 and H^1 norms
fprintf('\nOrder of Convergence:\n');
[l2,~] = regrese(h, l2errs);
[h1,~] = regrese(h, h1errs);
fprintf('  L^2-norm     ~ %.2f\n', l2);
fprintf('  H^1-seminorm ~ %.2f\n', h1);

end

function [p,q]=regrese(x,y)
% REGRESE Compute linear regression
% 
% Fits the line log10(y) = p log10(x) + q  in "loglog" coordinates 
%
% Returns: p, q  
%          p = estimated order of convergence

m=length(x);

A=[log10(x'), ones(m,1)];
r=A\log10(y');
p=r(1);
q=r(2);
end