function Z = basis_reduced_quad(p, X, Y)
%BASIS_REDUCED_QUAD Returns reduced quadrilateral basis evaluated at points
% in the reference square [-1,1]^2
%
% Only supports p=2,3.
%
% Arguments:
%     p - Polynomial degree of polynomial
%     X - Vector of x-coordinates of points to evaluate
%     Y - Vector of y-coordinates of points to evaluate
%
% Returns:
%     Nx(4p) matrix, where N is the number of points, containing per
%     row the values of the basis evaluated at the corresponding point.
%     The ordering of the basis points is as follows (this differs from the
%     lectures/practicals) - first the four vertices (as shown in the
%     diagram below); then all the basis for the first edge (E1)
%     left-to-right, then the second edge (E2) bottom-to-top, then E3
%     right-to-left, then E4 top-to-bottom (i.e., the all edge DoFs
%     counterclockwise from the first vertex).
%
%                 E3
%      (-1,1) 4--------3 (1,1)
%             |        |
%          E4 |        | E2
%             |        |
%     (-1,-1) 1--------2 (1,-1) 
%                 E1

Z = zeros(length(X), 4*p);
L = basis_lagrange_quad(p, X, Y);
if p == 2
    Z(:,1) = L(:,1)-L(:,9)/4;
    Z(:,2) = L(:,2)-L(:,9)/4;
    Z(:,3) = L(:,3)-L(:,9)/4;
    Z(:,4) = L(:,4)-L(:,9)/4;
    Z(:,5) = L(:,5)+L(:,9)/2;
    Z(:,6) = L(:,6)+L(:,9)/2;
    Z(:,7) = L(:,7)+L(:,9)/2;
    Z(:,8) = L(:,8)+L(:,9)/2;
elseif p == 3
    Z(:,1) = L(:,1)-(4*L(:,13)+2*L(:,14)+L(:,16)+2*L(:,15))/9;
    Z(:,2) = L(:,2)-(4*L(:,14)+2*L(:,16)+L(:,15)+2*L(:,13))/9;
    Z(:,3) = L(:,3)-(4*L(:,16)+2*L(:,15)+L(:,13)+2*L(:,14))/9;
    Z(:,4) = L(:,4)-(4*L(:,15)+2*L(:,13)+L(:,14)+2*L(:,16))/9;
    Z(:,5) = L(:,5)+(2*L(:,13)+L(:,15))/3;
    Z(:,6) = L(:,6)+(2*L(:,14)+L(:,16))/3;
    Z(:,7) = L(:,7)+(2*L(:,14)+L(:,13))/3;
    Z(:,8) = L(:,8)+(2*L(:,16)+L(:,15))/3;
    Z(:,9) = L(:,9)+(2*L(:,16)+L(:,14))/3;
    Z(:,10) = L(:,10)+(2*L(:,15)+L(:,13))/3;
    Z(:,11) = L(:,11)+(2*L(:,15)+L(:,16))/3;
    Z(:,12) = L(:,12)+(2*L(:,13)+L(:,14))/3;
else
    error('Reduced basis on rectangles must polynomial degree 2 or 3');
end
end