function Z = basis_lagrange_tri(p, X, Y)
%BASIS_LAGRANGE_TRI Returns total-order Lagrange basis evaluated at points
% in the reference triangle (-1,-1)-(1,-1)-(-1,1)
%
% Arguments:
%     p - Polynomial degree of Lagrange polynomial
%     X - Vector of x-coordinates of points to evaluate
%     Y - Vector of y-coordinates of points to evaluate
%
% Returns:
%     Nx((p+1)(p+2)/2) matrix, where N is the number of points, containing
%     per row the values of the basis evaluated at the corresponding point.
%     The ordering of the basis points is as follows (this differs from the
%     lectures/practicals) - first the three vertices (as shown in the
%     diagram below); then all the basis for the first edge (E1)
%     left-to-right, then the second edge (E2) bottom-to-top, then E3
%     top-to-bottom (i.e., all edge DoFs counterclockwise from the
%     first vertex); finally, the interior points
%     left-to-right on each row from the bottom row to the top
%
%      (-1,1) 3
%             |\
%             | \
%          E3 |  \ E2
%             |   \
%             |    \
%     (-1,-1) 1-----2 (1,-1) 
%                E1

if p < 1
    error('Lagrange basis on triangle must be at least polynomial degree 1');
end

lambda = zeros(length(X),3);
% Barycentric coordinates
lambda(:,1) = -(X+Y)/2;
lambda(:,2) = (1+X)/2;
lambda(:,3) = (1+Y)/2;

Z = zeros(length(X), (p+2)*(p+1)/2);

for idx=1:3
    % Vertex basis
    Z(:,idx) = lambda(:,idx);
    factor = 1;
    for j=1:(p-1)
        Z(:,idx) = Z(:,idx) .* (lambda(:,idx) - j/p);
        factor = factor * (1-j/p);
    end
    Z(:,idx) = Z(:,idx)/factor;
end
index = 4;
if p > 1
    % Edge basis
    for v1=1:3
        v2 = v1 + 1;
        if v2 > 3
            v2 = 1;
        end
        for idx=1:(p-1)
            Z(:,index) = lambda(:,v1).*lambda(:,v2);
            factor = idx*(p-idx)/(p^2);
            for k=1:(idx-1)
                Z(:,index) = Z(:,index) .* (lambda(:,v2) - k/p);
                factor = factor * (idx-k)/p;
            end
            for k=(idx+1):(p-1)
                Z(:,index) = Z(:,index) .* (lambda(:,v1) - (p-k)/p);
                factor = factor * (k-idx)/p;
            end
            Z(:,index) = Z(:,index)/factor;
            index = index+1;
        end
    end
end
if p > 2
    % Interior basis
    for j = 1:p-2
        for i = 1:p-1-j
            Z(:,index) = lambda(:,1).*lambda(:,2).*lambda(:,3);
            factor = i*j*(p-i-j)/(p^3);
            for k=1:i-1
                Z(:,index) = Z(:,index) .* (lambda(:,2)-k/p);
                factor = factor * (i-k)/p;
            end
            for k=1:j-1
                Z(:,index) = Z(:,index) .* (lambda(:,3)-k/p);
                factor = factor * (j-k)/p;
            end
            for k=(i+j+1):(p-1)
                Z(:,index) = Z(:,index) .* (lambda(:,1)-(p-k)/p);
                factor = factor * (k-i-j)/p;
            end
            Z(:,index) = Z(:,index)/factor;
            index = index+1;
        end
    end
end
end