function Z = basis_lagrange_quad(p, X, Y)
%BASIS_LAGRANGE_QUAD Returns tensor Lagrange basis evaluated at points
% in the reference square [-1,1]^2
%
% Arguments:
%     p - Polynomial degree of Lagrange polynomial
%     X - Vector of x-coordinates of points to evaluate
%     Y - Vector of y-coordinates of points to evaluate
%
% Returns:
%     Nx((p+1)^2) matrix, where N is the number of points, containing per
%     row the values of the basis evaluated at the corresponding point.
%     The ordering of the basis points is as follows (this differs from the
%     lectures/practicals) - first the four vertices (as shown in the
%     diagram below); then all the basis for the first edge (E1)
%     left-to-right, then the second edge (E2) bottom-to-top, then E3
%     right-to-left, then E4 top-to-bottom (i.e., all edge DoFs
%     counterclockwise from the first vertex); finally, the interior points
%     left-to-right on each row from the bottom row to the top
%
%                 E3
%      (-1,1) 4--------3 (1,1)
%             |        |
%          E4 |        | E2
%             |        |
%     (-1,-1) 1--------2 (1,-1) 
%                 E1

if p < 1
    error('Lagrange basis on rectangles must be at least polynomial degree 1');
end

% For quad we only need to tensor/cartesian product the 1D Lagrange basis
ZX = basis_lagrange_interval(p, X);
ZY = basis_lagrange_interval(p, Y);

Z = zeros(length(X), (p+1)^2);
% The four vertices first
Z(:,1) = ZX(:,1).*ZY(:,1);
Z(:,2) = ZX(:,p+1).*ZY(:,1);
Z(:,3) = ZX(:,p+1).*ZY(:,p+1);
Z(:,4) = ZX(:,1).*ZY(:,p+1);
if p > 1
    Z(:,5:(3+p)) = ZY(:,1).*ZX(:,2:p);              %E1
    Z(:,(4+p):(2+2*p)) = ZX(:,p+1).*ZY(:,2:p);      %E2
    Z(:,(3+2*p):(3*p+1)) = ZY(:,p+1).*ZX(:,p:-1:2); %E3
    Z(:,(2+3*p):(4*p)) = ZX(:,1).*ZY(:,p:-1:2);     %E4
    for i=2:p  % Loop over every row, bottom-to-top of interior DoFs
        % Row left-to-right
        Z(:,(5+(i+2)*(p-1)):(4+(i+3)*(p-1))) = ZX(:,2:p).*ZY(:,i);
    end
end
end